/*!
 * @nuxt/cli v2.15.8 (c) 2016-2021
 * Released under the MIT License
 * Repository: https://github.com/nuxt/nuxt.js
 * Website: https://nuxtjs.org
*/
'use strict';

const consola = require('consola');
const env = require('std-env');
const chalk = require('chalk');
const index = require('./cli-index.js');
const prettyBytes = require('pretty-bytes');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

const consola__default = /*#__PURE__*/_interopDefaultLegacy(consola);
const env__default = /*#__PURE__*/_interopDefaultLegacy(env);
const chalk__default = /*#__PURE__*/_interopDefaultLegacy(chalk);
const prettyBytes__default = /*#__PURE__*/_interopDefaultLegacy(prettyBytes);

function getMemoryUsage () {
  // https://nodejs.org/api/process.html#process_process_memoryusage
  const { heapUsed, rss } = process.memoryUsage();
  return { heap: heapUsed, rss }
}

function getFormattedMemoryUsage () {
  const { heap, rss } = getMemoryUsage();
  return `Memory usage: ${chalk__default['default'].bold(prettyBytes__default['default'](heap))} (RSS: ${prettyBytes__default['default'](rss)})`
}

function showMemoryUsage () {
  consola__default['default'].info(getFormattedMemoryUsage());
}

function showBanner (nuxt, showMemoryUsage = true) {
  if (env__default['default'].test) {
    return
  }

  if (env__default['default'].minimalCLI) {
    for (const listener of nuxt.server.listeners) {
      consola__default['default'].info('Listening on: ' + listener.url);
    }
    return
  }

  const titleLines = [];
  const messageLines = [];

  // Name and version
  const { bannerColor, badgeMessages } = nuxt.options.cli;
  titleLines.push(`${chalk__default['default'][bannerColor].bold('Nuxt')} @ ${nuxt.constructor.version || 'exotic'}\n`);

  const label = name => chalk__default['default'].bold.cyan(`▸ ${name}:`);

  // Environment
  const isDev = nuxt.options.dev;
  let _env = isDev ? 'development' : 'production';
  if (process.env.NODE_ENV !== _env) {
    _env += ` (${chalk__default['default'].cyan(process.env.NODE_ENV)})`;
  }
  titleLines.push(`${label('Environment')} ${_env}`);

  // Rendering
  const isSSR = nuxt.options.render.ssr;
  const rendering = isSSR ? 'server-side' : 'client-side';
  titleLines.push(`${label('Rendering')}   ${rendering}`);

  // Target
  const target = nuxt.options.target || 'server';
  titleLines.push(`${label('Target')}      ${target}`);

  if (showMemoryUsage) {
    titleLines.push('\n' + getFormattedMemoryUsage());
  }

  // Listeners
  for (const listener of nuxt.server.listeners) {
    messageLines.push(chalk__default['default'].bold('Listening: ') + chalk__default['default'].underline.blue(listener.url));
  }

  // Add custom badge messages
  if (badgeMessages.length) {
    messageLines.push('', ...badgeMessages);
  }

  process.stdout.write(index.successBox(messageLines.join('\n'), titleLines.join('\n')));
}

exports.showBanner = showBanner;
exports.showMemoryUsage = showMemoryUsage;
